<?php

declare(strict_types=1);

date_default_timezone_set('UTC');

function app_config(): array
{
    static $cfg = null;
    if ($cfg !== null) {
        return $cfg;
    }

    $defaults = [
        'app_name' => 'cTrader Bot Dashboard',
        'app_env' => 'production',
        'db' => [
            'host' => '127.0.0.1',
            'port' => '3306',
            'database' => 'derivbot',
            'username' => 'root',
            'password' => '',
            'charset' => 'utf8mb4',
        ],
        'security' => [
            'webhook_secret' => '',
            'signature_ttl_seconds' => 300,
        ],
        'dashboard' => [
            'offline_after_seconds' => 30,
            'event_retention_days' => 30,
        ],
    ];

    $configFile = dirname(__DIR__) . '/config.php';
    if (!is_file($configFile)) {
        throw new RuntimeException('Missing config.php. Create dashboard/config.php and set DB + security values.');
    }

    $userConfig = require $configFile;
    if (!is_array($userConfig)) {
        throw new RuntimeException('Invalid config.php. The file must return an array.');
    }

    $cfg = array_replace_recursive($defaults, $userConfig);

    return $cfg;
}

function db(): PDO
{
    static $pdo = null;
    if ($pdo instanceof PDO) {
        return $pdo;
    }

    $cfg = app_config()['db'];
    $dsn = sprintf(
        'mysql:host=%s;port=%s;dbname=%s;charset=%s',
        $cfg['host'],
        $cfg['port'],
        $cfg['database'],
        $cfg['charset']
    );

    $pdo = new PDO($dsn, $cfg['username'], $cfg['password'], [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ]);

    return $pdo;
}

function json_response(int $status, array $payload): void
{
    http_response_code($status);
    header('Content-Type: application/json');
    echo json_encode($payload, JSON_UNESCAPED_SLASHES);
    exit;
}

function now_utc(): string
{
    return gmdate('Y-m-d H:i:s');
}
