<?php
declare(strict_types=1);
require_once __DIR__ . '/src/bootstrap.php';
$cfg = app_config();
?>
<!doctype html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title><?= htmlspecialchars($cfg['app_name']) ?></title>
    <style>
        body { font-family: Arial, sans-serif; margin: 0; background: #0f172a; color: #e2e8f0; }
        .container { max-width: 1300px; margin: 0 auto; padding: 18px; }
        h1 { margin: 0 0 16px 0; font-size: 24px; }
        .grid { display: grid; gap: 12px; }
        .cards { grid-template-columns: repeat(auto-fit, minmax(220px, 1fr)); }
        .card { background: #111827; border: 1px solid #1f2937; border-radius: 10px; padding: 12px 14px; }
        .label { font-size: 12px; color: #94a3b8; }
        .value { font-size: 24px; font-weight: 700; margin-top: 5px; }
        .row { margin-top: 14px; display: grid; grid-template-columns: 1fr 1fr; gap: 12px; }
        .panel { background: #111827; border: 1px solid #1f2937; border-radius: 10px; padding: 12px; }
        .panel h2 { margin: 0 0 10px 0; font-size: 16px; }
        table { width: 100%; border-collapse: collapse; font-size: 13px; }
        th, td { border-bottom: 1px solid #1f2937; text-align: left; padding: 8px 6px; vertical-align: top; }
        th { color: #94a3b8; font-weight: 600; }
        .controls { display: flex; flex-wrap: wrap; gap: 8px; margin-bottom: 10px; }
        input, select, button { background: #0b1220; color: #e2e8f0; border: 1px solid #334155; border-radius: 6px; padding: 6px 8px; }
        button { cursor: pointer; }
        .pill { border-radius: 999px; font-size: 11px; padding: 2px 8px; display: inline-block; }
        .pill.online { background: #14532d; color: #86efac; }
        .pill.offline { background: #7f1d1d; color: #fecaca; }
        .small { font-size: 12px; color: #94a3b8; }
        @media (max-width: 980px) { .row { grid-template-columns: 1fr; } }
    </style>
</head>
<body>
<div class="container">
    <h1><?= htmlspecialchars($cfg['app_name']) ?></h1>
    <div class="small">Auto-refresh every 5 seconds. API: <code>/dashboard/api</code></div>

    <div class="grid cards" style="margin-top: 12px;">
        <div class="card"><div class="label">Online Bots</div><div id="cardOnlineBots" class="value">0</div></div>
        <div class="card"><div class="label">Open Trades</div><div id="cardOpenTrades" class="value">0</div></div>
        <div class="card"><div class="label">Closed Today</div><div id="cardClosedToday" class="value">0</div></div>
        <div class="card"><div class="label">Net PnL</div><div id="cardNetPnl" class="value">$0.00</div></div>
    </div>

    <div class="row">
        <div class="panel">
            <h2>Bot Status</h2>
            <table>
                <thead><tr><th>Bot</th><th>Status</th><th>Last Seen</th><th>Last Sequence</th></tr></thead>
                <tbody id="botsTable"></tbody>
            </table>
        </div>
        <div class="panel">
            <h2>Latest Snapshot</h2>
            <table>
                <tbody id="latestSnapshotTable"></tbody>
            </table>
        </div>
    </div>

    <div class="panel" style="margin-top: 12px;">
        <h2>Recent Events</h2>
        <div class="controls">
            <input id="eventFilterSymbol" placeholder="Symbol filter">
            <select id="eventFilterType"><option value="">All event types</option></select>
            <input id="eventDateFrom" type="datetime-local">
            <input id="eventDateTo" type="datetime-local">
            <button id="eventApplyBtn">Apply</button>
        </div>
        <table>
            <thead><tr><th>Time</th><th>Type</th><th>Severity</th><th>Symbol</th><th>Message</th></tr></thead>
            <tbody id="eventsTable"></tbody>
        </table>
    </div>

    <div class="row">
        <div class="panel">
            <h2>Snapshots</h2>
            <table>
                <thead><tr><th>Time</th><th>Symbol</th><th>Stake</th><th>BuyTrend</th><th>SellTrend</th><th>OpenPos</th></tr></thead>
                <tbody id="snapshotsTable"></tbody>
            </table>
        </div>
        <div class="panel">
            <h2>Trades</h2>
            <div class="controls">
                <input id="tradeFilterSymbol" placeholder="Symbol filter">
                <input id="tradeDateFrom" type="datetime-local">
                <input id="tradeDateTo" type="datetime-local">
                <button id="tradeApplyBtn">Apply</button>
            </div>
            <table>
                <thead><tr><th>Closed</th><th>Symbol</th><th>Side</th><th>Volume</th><th>PnL</th></tr></thead>
                <tbody id="tradesTable"></tbody>
            </table>
        </div>
    </div>
</div>

<script>
const API_BASE = '/dashboard/api';
let cachedEvents = [];
let cachedTrades = [];

const fmtMoney = n => `$${Number(n || 0).toFixed(2)}`;
const toTs = s => s ? new Date(s).getTime() : null;
const esc = s => String(s ?? '').replace(/[&<>"]/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;'}[c]));

async function getJson(url) {
    const r = await fetch(url);
    return r.json();
}

function renderOverview(data) {
    const online = (data.bots || []).filter(b => b.is_online).length;
    document.getElementById('cardOnlineBots').textContent = online;
    document.getElementById('cardOpenTrades').textContent = data.open_trades ?? 0;
    document.getElementById('cardClosedToday').textContent = data.closed_today ?? 0;
    document.getElementById('cardNetPnl').textContent = fmtMoney(data.net_pnl ?? 0);

    const botsRows = (data.bots || []).map(b => `
        <tr>
            <td>${esc(b.bot_id)}</td>
            <td><span class="pill ${b.is_online ? 'online' : 'offline'}">${b.is_online ? 'online' : 'offline'}</span></td>
            <td>${esc(b.last_seen_at)}</td>
            <td>${esc(b.last_sequence)}</td>
        </tr>`).join('');
    document.getElementById('botsTable').innerHTML = botsRows || `<tr><td colspan="4">No bot data yet</td></tr>`;

    const s = data.latest_snapshot;
    const html = s ? `
        <tr><th>Symbol</th><td>${esc(s.symbol)}</td></tr>
        <tr><th>Stake</th><td>${esc(s.stake)} (index ${esc(s.stake_index)})</td></tr>
        <tr><th>Thresholds</th><td>Hedge ${esc(s.hedge_loss_threshold)} | Trailing ${esc(s.trailing_stop_threshold)}</td></tr>
        <tr><th>Max Profit</th><td>Buy ${esc(s.max_buy_profit)} | Sell ${esc(s.max_sell_profit)}</td></tr>
        <tr><th>Trends</th><td>Buy ${s.is_buy_trend ? 'true':'false'} | Sell ${s.is_sell_trend ? 'true':'false'}</td></tr>
        <tr><th>Open Positions</th><td>${esc(s.open_positions_count)}</td></tr>
        <tr><th>Snapshot Time</th><td>${esc(s.sent_at_utc)}</td></tr>
    ` : `<tr><td>No snapshots yet</td></tr>`;
    document.getElementById('latestSnapshotTable').innerHTML = html;
}

function renderEvents(events) {
    const symbol = document.getElementById('eventFilterSymbol').value.trim().toLowerCase();
    const type = document.getElementById('eventFilterType').value.trim();
    const from = toTs(document.getElementById('eventDateFrom').value);
    const to = toTs(document.getElementById('eventDateTo').value);

    const rows = events.filter(e => {
        const t = toTs(e.sent_at_utc);
        if (symbol && !(e.symbol || '').toLowerCase().includes(symbol)) return false;
        if (type && e.event_type !== type) return false;
        if (from && t < from) return false;
        if (to && t > to) return false;
        return true;
    }).map(e => `
        <tr>
            <td>${esc(e.sent_at_utc)}</td>
            <td>${esc(e.event_type)}</td>
            <td>${esc(e.severity)}</td>
            <td>${esc(e.symbol || '')}</td>
            <td>${esc(e.message)}</td>
        </tr>`).join('');
    document.getElementById('eventsTable').innerHTML = rows || `<tr><td colspan="5">No events matched filters</td></tr>`;
}

function renderEventTypeOptions(events) {
    const types = [...new Set(events.map(e => e.event_type).filter(Boolean))].sort();
    const sel = document.getElementById('eventFilterType');
    const prev = sel.value;
    sel.innerHTML = `<option value="">All event types</option>` + types.map(t => `<option value="${esc(t)}">${esc(t)}</option>`).join('');
    sel.value = prev;
}

function renderSnapshots(snaps) {
    const rows = snaps.map(s => `
        <tr>
            <td>${esc(s.sent_at_utc)}</td>
            <td>${esc(s.symbol)}</td>
            <td>${esc(s.stake)}</td>
            <td>${s.is_buy_trend ? 'true' : 'false'}</td>
            <td>${s.is_sell_trend ? 'true' : 'false'}</td>
            <td>${esc(s.open_positions_count)}</td>
        </tr>`).join('');
    document.getElementById('snapshotsTable').innerHTML = rows || `<tr><td colspan="6">No snapshots yet</td></tr>`;
}

function renderTrades(trades) {
    const symbol = document.getElementById('tradeFilterSymbol').value.trim().toLowerCase();
    const from = toTs(document.getElementById('tradeDateFrom').value);
    const to = toTs(document.getElementById('tradeDateTo').value);
    const rows = trades.filter(t => {
        const x = toTs(t.closed_at || t.opened_at);
        if (symbol && !(t.symbol || '').toLowerCase().includes(symbol)) return false;
        if (from && x < from) return false;
        if (to && x > to) return false;
        return true;
    }).map(t => `
        <tr>
            <td>${esc(t.closed_at || t.opened_at || '')}</td>
            <td>${esc(t.symbol || '')}</td>
            <td>${esc(t.side || '')}</td>
            <td>${esc(t.volume || '')}</td>
            <td>${fmtMoney(t.gross_profit)}</td>
        </tr>`).join('');
    document.getElementById('tradesTable').innerHTML = rows || `<tr><td colspan="5">No trades matched filters</td></tr>`;
}

async function refresh() {
    try {
        const [overview, events, snapshots, trades] = await Promise.all([
            getJson(`${API_BASE}/dashboard/overview`),
            getJson(`${API_BASE}/dashboard/events?limit=100`),
            getJson(`${API_BASE}/dashboard/snapshots?limit=100`),
            getJson(`${API_BASE}/dashboard/trades?limit=100`)
        ]);
        if (overview.ok) renderOverview(overview.data);
        if (events.ok) {
            cachedEvents = events.data || [];
            renderEventTypeOptions(cachedEvents);
            renderEvents(cachedEvents);
        }
        if (snapshots.ok) renderSnapshots(snapshots.data || []);
        if (trades.ok) {
            cachedTrades = trades.data || [];
            renderTrades(cachedTrades);
        }
    } catch (err) {
        console.error(err);
    }
}

document.getElementById('eventApplyBtn').addEventListener('click', () => renderEvents(cachedEvents));
document.getElementById('tradeApplyBtn').addEventListener('click', () => renderTrades(cachedTrades));
document.getElementById('eventFilterType').addEventListener('change', () => renderEvents(cachedEvents));

refresh();
setInterval(refresh, 5000);
</script>
</body>
</html>
