<?php

declare(strict_types=1);

require_once __DIR__ . '/../src/bootstrap.php';

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$path = '/';
$explicitEndpoint = trim((string) ($_GET['endpoint'] ?? ''), '/');
if ($explicitEndpoint !== '') {
    // No-rewrite mode: /dashboard/api/index.php?endpoint=telemetry/heartbeat
    $path = '/' . $explicitEndpoint;
} else {
    // Rewrite mode fallback (if enabled)
    $path = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH) ?? '/';
    $path = preg_replace('#^/+#', '/', $path);
    if (str_starts_with($path, '/dashboard/api/')) {
        $path = substr($path, strlen('/dashboard/api'));
    } elseif (str_starts_with($path, '/api/')) {
        $path = substr($path, strlen('/api'));
    }
    if ($path === false || $path === '' || $path === '/index.php' || $path === '/dashboard/api/index.php') {
        $path = '/';
    }
}

try {
    if ($method === 'GET' && ($path === '/' || $path === '/telemetry')) {
        json_response(200, [
            'ok' => true,
            'mode' => $explicitEndpoint !== '' ? 'no_rewrite' : 'rewrite',
            'message' => 'API is running',
            'usage' => [
                'no_rewrite_post_example' => '/dashboard/api/index.php?endpoint=telemetry/heartbeat',
                'no_rewrite_get_example' => '/dashboard/api/index.php?endpoint=dashboard/overview',
            ],
        ]);
    }

    if ($method === 'POST' && $path === '/telemetry/event') {
        verify_telemetry_signature();
        $envelope = parse_envelope();
        ingest_event($envelope);
        json_response(200, ['ok' => true, 'ingested_at_utc' => gmdate('c')]);
    }

    if ($method === 'POST' && $path === '/telemetry/snapshot') {
        verify_telemetry_signature();
        $envelope = parse_envelope();
        ingest_snapshot($envelope);
        json_response(200, ['ok' => true, 'ingested_at_utc' => gmdate('c')]);
    }

    if ($method === 'POST' && $path === '/telemetry/heartbeat') {
        verify_telemetry_signature();
        $envelope = parse_envelope();
        ingest_heartbeat($envelope);
        json_response(200, ['ok' => true, 'ingested_at_utc' => gmdate('c')]);
    }

    if ($method === 'GET' && $path === '/dashboard/overview') {
        json_response(200, ['ok' => true, 'data' => dashboard_overview()]);
    }

    if ($method === 'GET' && $path === '/dashboard/events') {
        $limit = max(1, min(200, (int) ($_GET['limit'] ?? 50)));
        json_response(200, ['ok' => true, 'data' => dashboard_events($limit)]);
    }

    if ($method === 'GET' && $path === '/dashboard/snapshots') {
        $limit = max(1, min(200, (int) ($_GET['limit'] ?? 50)));
        json_response(200, ['ok' => true, 'data' => dashboard_snapshots($limit)]);
    }

    if ($method === 'GET' && $path === '/dashboard/trades') {
        $limit = max(1, min(200, (int) ($_GET['limit'] ?? 50)));
        json_response(200, ['ok' => true, 'data' => dashboard_trades($limit)]);
    }

    json_response(404, ['ok' => false, 'error' => 'not_found']);
} catch (RuntimeException $e) {
    json_response(400, ['ok' => false, 'error' => $e->getMessage()]);
} catch (Throwable $e) {
    json_response(500, ['ok' => false, 'error' => 'server_error', 'detail' => $e->getMessage()]);
}

function parse_envelope(): array
{
    $raw = file_get_contents('php://input');
    if (!is_string($raw) || $raw === '') {
        throw new RuntimeException('empty_body');
    }

    $data = json_decode($raw, true);
    if (!is_array($data)) {
        throw new RuntimeException('invalid_json');
    }

    foreach (['schema_version', 'bot_id', 'sequence', 'sent_at_utc', 'payload'] as $required) {
        if (!array_key_exists($required, $data)) {
            throw new RuntimeException('invalid_envelope');
        }
    }

    if ((int) $data['schema_version'] !== 1) {
        throw new RuntimeException('unsupported_schema');
    }

    if (!is_array($data['payload'])) {
        throw new RuntimeException('invalid_payload');
    }

    return $data;
}

function verify_telemetry_signature(): void
{
    $cfg = app_config()['security'];
    $raw = file_get_contents('php://input');
    if (!is_string($raw)) {
        throw new RuntimeException('invalid_body');
    }

    $timestamp = $_SERVER['HTTP_X_TIMESTAMP'] ?? '';
    $nonce = $_SERVER['HTTP_X_NONCE'] ?? '';
    $signature = strtolower($_SERVER['HTTP_X_SIGNATURE'] ?? '');

    if ($timestamp === '' || $nonce === '' || $signature === '') {
        throw new RuntimeException('missing_auth_headers');
    }

    $now = time();
    $ts = (int) $timestamp;
    $ttl = (int) $cfg['signature_ttl_seconds'];
    if ($ts <= 0 || abs($now - $ts) > $ttl) {
        throw new RuntimeException('timestamp_out_of_window');
    }

    remember_nonce_or_fail('telemetry', $nonce);

    $toSign = $timestamp . "\n" . $nonce . "\n" . $raw;
    $expected = hash_hmac('sha256', $toSign, (string) $cfg['webhook_secret']);
    if (!hash_equals($expected, $signature)) {
        throw new RuntimeException('invalid_signature');
    }
}

function remember_nonce_or_fail(string $apiKey, string $nonce): void
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO api_nonces (api_key, nonce, created_at) VALUES (:api_key, :nonce, :created_at)');
    try {
        $stmt->execute([
            ':api_key' => $apiKey,
            ':nonce' => $nonce,
            ':created_at' => now_utc(),
        ]);
    } catch (Throwable $e) {
        throw new RuntimeException('replay_detected');
    }
}

function upsert_bot(string $botId, int $sequence, string $status = 'online'): void
{
    $sql = 'INSERT INTO bots (bot_id, status, last_seen_at, last_sequence, created_at, updated_at)
            VALUES (:bot_id, :status, :seen, :seq, :created, :updated)
            ON DUPLICATE KEY UPDATE
                status = VALUES(status),
                last_seen_at = VALUES(last_seen_at),
                last_sequence = GREATEST(last_sequence, VALUES(last_sequence)),
                updated_at = VALUES(updated_at)';
    $stmt = db()->prepare($sql);
    $now = now_utc();
    $stmt->execute([
        ':bot_id' => $botId,
        ':status' => $status,
        ':seen' => $now,
        ':seq' => $sequence,
        ':created' => $now,
        ':updated' => $now,
    ]);
}

function ingest_event(array $envelope): void
{
    $botId = (string) $envelope['bot_id'];
    $sequence = (int) $envelope['sequence'];
    $sentAt = gmdate('Y-m-d H:i:s', strtotime((string) $envelope['sent_at_utc']));
    $payload = $envelope['payload'];

    upsert_bot($botId, $sequence, 'online');

    $stmt = db()->prepare(
        'INSERT INTO bot_events
        (bot_id, sequence, event_type, severity, symbol, message, trade_json, metrics_json, context_json, payload_json, sent_at_utc, created_at)
        VALUES
        (:bot_id, :sequence, :event_type, :severity, :symbol, :message, :trade_json, :metrics_json, :context_json, :payload_json, :sent_at_utc, :created_at)'
    );

    $stmt->execute([
        ':bot_id' => $botId,
        ':sequence' => $sequence,
        ':event_type' => (string) ($payload['event_type'] ?? 'unknown'),
        ':severity' => (string) ($payload['severity'] ?? 'info'),
        ':symbol' => isset($payload['symbol']) ? (string) $payload['symbol'] : null,
        ':message' => (string) ($payload['message'] ?? ''),
        ':trade_json' => isset($payload['trade']) ? json_encode($payload['trade'], JSON_UNESCAPED_SLASHES) : null,
        ':metrics_json' => isset($payload['metrics']) ? json_encode($payload['metrics'], JSON_UNESCAPED_SLASHES) : null,
        ':context_json' => isset($payload['context']) ? json_encode($payload['context'], JSON_UNESCAPED_SLASHES) : null,
        ':payload_json' => json_encode($payload, JSON_UNESCAPED_SLASHES),
        ':sent_at_utc' => $sentAt,
        ':created_at' => now_utc(),
    ]);

    $eventId = (int) db()->lastInsertId();
    maybe_upsert_trade_from_event($botId, $payload, $eventId, $sentAt);
}

function maybe_upsert_trade_from_event(string $botId, array $payload, int $eventId, string $sentAt): void
{
    $eventType = (string) ($payload['event_type'] ?? '');
    if (!in_array($eventType, ['trade_opened', 'trade_closed'], true)) {
        return;
    }

    $trade = $payload['trade'] ?? null;
    if (!is_array($trade) || empty($trade['position_id'])) {
        return;
    }

    $positionId = (string) $trade['position_id'];
    $symbol = isset($payload['symbol']) ? (string) $payload['symbol'] : null;

    if ($eventType === 'trade_opened') {
        $sql = 'INSERT INTO trades
                (bot_id, symbol, position_id, label, side, volume, entry_price, opened_at, open_event_id, created_at, updated_at)
                VALUES
                (:bot_id, :symbol, :position_id, :label, :side, :volume, :entry_price, :opened_at, :open_event_id, :created_at, :updated_at)
                ON DUPLICATE KEY UPDATE
                  symbol = VALUES(symbol),
                  label = VALUES(label),
                  side = VALUES(side),
                  volume = VALUES(volume),
                  entry_price = VALUES(entry_price),
                  opened_at = COALESCE(opened_at, VALUES(opened_at)),
                  open_event_id = COALESCE(open_event_id, VALUES(open_event_id)),
                  updated_at = VALUES(updated_at)';
        $stmt = db()->prepare($sql);
        $stmt->execute([
            ':bot_id' => $botId,
            ':symbol' => $symbol,
            ':position_id' => $positionId,
            ':label' => isset($trade['label']) ? (string) $trade['label'] : null,
            ':side' => isset($trade['side']) ? (string) $trade['side'] : null,
            ':volume' => isset($trade['volume']) ? (float) $trade['volume'] : null,
            ':entry_price' => isset($trade['entry_price']) ? (float) $trade['entry_price'] : null,
            ':opened_at' => $sentAt,
            ':open_event_id' => $eventId,
            ':created_at' => now_utc(),
            ':updated_at' => now_utc(),
        ]);
    } else {
        $sql = 'INSERT INTO trades
                (bot_id, symbol, position_id, label, side, volume, exit_price, gross_profit, closed_at, close_event_id, created_at, updated_at)
                VALUES
                (:bot_id, :symbol, :position_id, :label, :side, :volume, :exit_price, :gross_profit, :closed_at, :close_event_id, :created_at, :updated_at)
                ON DUPLICATE KEY UPDATE
                  symbol = VALUES(symbol),
                  label = COALESCE(VALUES(label), label),
                  side = COALESCE(VALUES(side), side),
                  volume = COALESCE(VALUES(volume), volume),
                  exit_price = VALUES(exit_price),
                  gross_profit = VALUES(gross_profit),
                  closed_at = VALUES(closed_at),
                  close_event_id = VALUES(close_event_id),
                  updated_at = VALUES(updated_at)';
        $stmt = db()->prepare($sql);
        $stmt->execute([
            ':bot_id' => $botId,
            ':symbol' => $symbol,
            ':position_id' => $positionId,
            ':label' => isset($trade['label']) ? (string) $trade['label'] : null,
            ':side' => isset($trade['side']) ? (string) $trade['side'] : null,
            ':volume' => isset($trade['volume']) ? (float) $trade['volume'] : null,
            ':exit_price' => isset($trade['exit_price']) ? (float) $trade['exit_price'] : null,
            ':gross_profit' => isset($trade['gross_profit']) ? (float) $trade['gross_profit'] : null,
            ':closed_at' => $sentAt,
            ':close_event_id' => $eventId,
            ':created_at' => now_utc(),
            ':updated_at' => now_utc(),
        ]);
    }
}

function ingest_snapshot(array $envelope): void
{
    $botId = (string) $envelope['bot_id'];
    $sequence = (int) $envelope['sequence'];
    $payload = $envelope['payload'];
    $sentAt = gmdate('Y-m-d H:i:s', strtotime((string) $envelope['sent_at_utc']));

    upsert_bot($botId, $sequence, 'online');

    $stmt = db()->prepare(
        'INSERT INTO bot_state_snapshots
        (bot_id, symbol, sequence, stake_index, stake, hedge_loss_threshold, trailing_stop_threshold, last_trade_time_utc, restart_time_utc, max_buy_profit, max_sell_profit, is_buy_trend, is_sell_trend, open_positions_count, open_positions_json, indicators_json, payload_json, sent_at_utc, created_at)
        VALUES
        (:bot_id, :symbol, :sequence, :stake_index, :stake, :hedge_loss_threshold, :trailing_stop_threshold, :last_trade_time_utc, :restart_time_utc, :max_buy_profit, :max_sell_profit, :is_buy_trend, :is_sell_trend, :open_positions_count, :open_positions_json, :indicators_json, :payload_json, :sent_at_utc, :created_at)'
    );

    $stmt->execute([
        ':bot_id' => $botId,
        ':symbol' => (string) ($payload['symbol'] ?? 'unknown'),
        ':sequence' => $sequence,
        ':stake_index' => (int) ($payload['stake_index'] ?? 0),
        ':stake' => (float) ($payload['stake'] ?? 0),
        ':hedge_loss_threshold' => (float) ($payload['hedge_loss_threshold'] ?? 0),
        ':trailing_stop_threshold' => (float) ($payload['trailing_stop_threshold'] ?? 0),
        ':last_trade_time_utc' => normalize_datetime_or_null($payload['last_trade_time_utc'] ?? null),
        ':restart_time_utc' => normalize_datetime_or_null($payload['restart_time_utc'] ?? null),
        ':max_buy_profit' => (float) ($payload['max_buy_profit'] ?? 0),
        ':max_sell_profit' => (float) ($payload['max_sell_profit'] ?? 0),
        ':is_buy_trend' => !empty($payload['is_buy_trend']) ? 1 : 0,
        ':is_sell_trend' => !empty($payload['is_sell_trend']) ? 1 : 0,
        ':open_positions_count' => (int) ($payload['open_positions_count'] ?? 0),
        ':open_positions_json' => isset($payload['open_positions']) ? json_encode($payload['open_positions'], JSON_UNESCAPED_SLASHES) : null,
        ':indicators_json' => isset($payload['latest_indicators']) ? json_encode($payload['latest_indicators'], JSON_UNESCAPED_SLASHES) : null,
        ':payload_json' => json_encode($payload, JSON_UNESCAPED_SLASHES),
        ':sent_at_utc' => $sentAt,
        ':created_at' => now_utc(),
    ]);
}

function ingest_heartbeat(array $envelope): void
{
    $botId = (string) $envelope['bot_id'];
    $sequence = (int) $envelope['sequence'];
    $payload = $envelope['payload'];
    $status = (string) ($payload['status'] ?? 'online');
    upsert_bot($botId, $sequence, $status);
}

function normalize_datetime_or_null($value): ?string
{
    if ($value === null || $value === '') {
        return null;
    }
    $ts = strtotime((string) $value);
    if ($ts === false) {
        return null;
    }
    return gmdate('Y-m-d H:i:s', $ts);
}

function dashboard_overview(): array
{
    $pdo = db();
    $offlineAfter = (int) app_config()['dashboard']['offline_after_seconds'];

    $bots = $pdo->query('SELECT bot_id, status, last_seen_at, last_sequence FROM bots ORDER BY last_seen_at DESC')->fetchAll();
    foreach ($bots as &$bot) {
        $bot['is_online'] = (time() - strtotime((string) $bot['last_seen_at'])) <= $offlineAfter;
    }

    $openTrades = (int) $pdo->query('SELECT COUNT(*) FROM trades WHERE closed_at IS NULL')->fetchColumn();
    $netPnl = (float) $pdo->query('SELECT COALESCE(SUM(gross_profit), 0) FROM trades WHERE closed_at IS NOT NULL')->fetchColumn();
    $closedToday = (int) $pdo->query("SELECT COUNT(*) FROM trades WHERE closed_at >= UTC_DATE()")->fetchColumn();
    $latestSnapshot = $pdo->query('SELECT * FROM bot_state_snapshots ORDER BY created_at DESC LIMIT 1')->fetch();

    return [
        'bots' => $bots,
        'open_trades' => $openTrades,
        'closed_today' => $closedToday,
        'net_pnl' => round($netPnl, 2),
        'latest_snapshot' => $latestSnapshot ?: null,
    ];
}

function dashboard_events(int $limit): array
{
    $stmt = db()->prepare(
        'SELECT id, bot_id, event_type, severity, symbol, message, trade_json, metrics_json, context_json, sent_at_utc, created_at
         FROM bot_events ORDER BY id DESC LIMIT :limit'
    );
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

function dashboard_snapshots(int $limit): array
{
    $stmt = db()->prepare(
        'SELECT id, bot_id, symbol, stake_index, stake, hedge_loss_threshold, trailing_stop_threshold,
                max_buy_profit, max_sell_profit, is_buy_trend, is_sell_trend, open_positions_count,
                open_positions_json, indicators_json, sent_at_utc, created_at
         FROM bot_state_snapshots ORDER BY id DESC LIMIT :limit'
    );
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

function dashboard_trades(int $limit): array
{
    $stmt = db()->prepare(
        'SELECT id, bot_id, symbol, position_id, label, side, volume, entry_price, exit_price, gross_profit, opened_at, closed_at
         FROM trades ORDER BY id DESC LIMIT :limit'
    );
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}
